<?php
// home.php - Agent home page with call history
require_once 'config.php';
requireLogin();
regenerateSession();

$agent_phone = getAgentPhone();
$agent_name = $_SESSION['agent_name'] ?? 'Agent';

// Pagination
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$limit = 15;
$offset = ($page - 1) * $limit;

// Filters
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$caller_number = sanitizeInput($_GET['caller_number'] ?? '');

$conn = getDBConnection();

// Build query with filters
$where_conditions = ["agent_contact = ?"];
$params = [$agent_phone];
$types = "s";

if (!empty($date_from)) {
    $where_conditions[] = "DATE(call_date) >= ?";
    $params[] = $date_from;
    $types .= "s";
}

if (!empty($date_to)) {
    $where_conditions[] = "DATE(call_date) <= ?";
    $params[] = $date_to;
    $types .= "s";
}

if (!empty($caller_number)) {
    $where_conditions[] = "client_contact LIKE ?";
    $params[] = "%$caller_number%";
    $types .= "s";
}

$where_clause = implode(" AND ", $where_conditions);

// Get total count
$count_sql = "SELECT COUNT(*) as total FROM call_logs WHERE $where_clause";
$count_stmt = $conn->prepare($count_sql);
$count_stmt->bind_param($types, ...$params);
$count_stmt->execute();
$total_calls = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_calls / $limit);
$count_stmt->close();

// Get calls for current page
$sql = "SELECT id, call_date, client_contact, duration, status 
        FROM call_logs 
        WHERE $where_clause 
        ORDER BY call_date DESC 
        LIMIT ? OFFSET ?";

$stmt = $conn->prepare($sql);
$params[] = $limit;
$params[] = $offset;
$types .= "ii";
$stmt->bind_param($types, ...$params);
$stmt->execute();
$result = $stmt->get_result();
$calls = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Get uncommented calls count for badge
$badge_stmt = $conn->prepare("
    SELECT COUNT(*) as count 
    FROM call_logs cl 
    LEFT JOIN call_notes cn ON cl.id = cn.call_id 
    WHERE cl.agent_contact = ? AND cn.id IS NULL
");
$badge_stmt->bind_param("s", $agent_phone);
$badge_stmt->execute();
$uncommented_count = $badge_stmt->get_result()->fetch_assoc()['count'];
$badge_stmt->close();

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Home - Agent Portal</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div class="header">
        <div class="header-content">
            <h1>Call History</h1>
            <a href="logout.php" class="logout-btn">Logout</a>
        </div>
    </div>

    <div class="container">
        <!-- Filter Section -->
        <div class="filter-section">
            <h3>Filter Calls</h3>
            <form method="GET" action="home.php">
                <div class="filter-group">
                    <label for="date_from">From Date</label>
                    <input type="date" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                </div>
                
                <div class="filter-group">
                    <label for="date_to">To Date</label>
                    <input type="date" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                </div>
                
                <div class="filter-group">
                    <label for="caller_number">Caller Number</label>
                    <input type="tel" id="caller_number" name="caller_number" placeholder="Search by number" value="<?php echo htmlspecialchars($caller_number); ?>">
                </div>
                
                <div class="filter-buttons">
                    <button type="submit" class="btn-filter">Apply Filter</button>
                    <a href="home.php" class="btn-clear" style="text-align: center; line-height: 38px; text-decoration: none;">Clear</a>
                </div>
            </form>
        </div>

        <!-- Calls Table -->
        <?php if (count($calls) > 0): ?>
            <div class="calls-table-wrapper">
                <table class="calls-table">
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>Date & Time</th>
                            <th>Caller</th>
                            <th>Duration</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($calls as $index => $call): 
                            $row_number = $offset + $index + 1;
                            $duration_mins = floor($call['duration'] / 60);
                            $duration_secs = $call['duration'] % 60;
                        ?>
                            <tr>
                                <td><?php echo $row_number; ?></td>
                                <td><?php echo date('M d, Y H:i', strtotime($call['call_date'])); ?></td>
                                <td>
                                    <a href="tel:<?php echo htmlspecialchars($call['client_contact']); ?>" class="tel-link">
                                        <?php echo htmlspecialchars($call['client_contact']); ?>
                                    </a>
                                </td>
                                <td><?php echo sprintf("%02d:%02d", $duration_mins, $duration_secs); ?></td>
                                <td><?php echo ucfirst($call['status']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
                <div class="pagination">
                    <button 
                        onclick="window.location.href='?page=<?php echo $page - 1; ?><?php echo $date_from ? '&date_from='.$date_from : ''; ?><?php echo $date_to ? '&date_to='.$date_to : ''; ?><?php echo $caller_number ? '&caller_number='.urlencode($caller_number) : ''; ?>'"
                        <?php echo $page <= 1 ? 'disabled' : ''; ?>
                    >
                        Previous
                    </button>
                    <span>Page <?php echo $page; ?> of <?php echo $total_pages; ?></span>
                    <button 
                        onclick="window.location.href='?page=<?php echo $page + 1; ?><?php echo $date_from ? '&date_from='.$date_from : ''; ?><?php echo $date_to ? '&date_to='.$date_to : ''; ?><?php echo $caller_number ? '&caller_number='.urlencode($caller_number) : ''; ?>'"
                        <?php echo $page >= $total_pages ? 'disabled' : ''; ?>
                    >
                        Next
                    </button>
                </div>
            <?php endif; ?>
        <?php else: ?>
            <div class="empty-state">
                <div class="empty-state-icon">📞</div>
                <p>No calls found</p>
            </div>
        <?php endif; ?>
    </div>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <a href="home.php" class="nav-item active">
            <div class="nav-icon">🏠</div>
            <div class="nav-label">Home</div>
        </a>
        <a href="calls.php" class="nav-item">
            <div class="nav-icon">
                📋
                <?php if ($uncommented_count > 0): ?>
                    <span class="nav-badge"><?php echo $uncommented_count; ?></span>
                <?php endif; ?>
            </div>
            <div class="nav-label">Calls</div>
        </a>
        <a href="profile.php" class="nav-item">
            <div class="nav-icon">👤</div>
            <div class="nav-label">Profile</div>
        </a>
    </nav>

    <script>
        // Mobile viewport check
        function checkViewport() {
            if (window.innerWidth > 756) {
                document.body.innerHTML = `
                    <div class="error-404">
                        <h1>404</h1>
                        <p>This application is only accessible on mobile devices.</p>
                    </div>
                `;
            }
        }
        
        checkViewport();
        window.addEventListener('resize', checkViewport);

        // Auto-refresh for new calls (every 30 seconds)
        setInterval(() => {
            if (!document.hidden) {
                location.reload();
            }
        }, 30000);
    </script>
</body>
</html>