<?php
/**
 * set_agent_password.php
 * Helper script to set passwords for agents
 * 
 * Usage: Run this script from command line or browser (remove after use for security)
 * php set_agent_password.php
 */

require_once 'config.php';

// SECURITY: Remove or restrict access to this file after setting up passwords
// You can add IP restriction or require authentication here

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Set Agent Password</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div class="login-container">
        <div class="login-box">
            <h1>Set Agent Password</h1>
            <p>Set or update agent passwords</p>
            
            <?php
            $message = '';
            $error = '';
            
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $contact = sanitizeInput($_POST['contact'] ?? '');
                $password = $_POST['password'] ?? '';
                $confirm_password = $_POST['confirm_password'] ?? '';
                
                if (empty($contact) || empty($password)) {
                    $error = 'Please provide contact and password.';
                } elseif ($password !== $confirm_password) {
                    $error = 'Passwords do not match.';
                } elseif (strlen($password) < 6) {
                    $error = 'Password must be at least 6 characters long.';
                } else {
                    $conn = getDBConnection();
                    
                    // Check if agent exists
                    $check_stmt = $conn->prepare("SELECT id, name FROM agents WHERE contact = ?");
                    $check_stmt->bind_param("s", $contact);
                    $check_stmt->execute();
                    $result = $check_stmt->get_result();
                    
                    if ($result->num_rows === 1) {
                        $agent = $result->fetch_assoc();
                        $check_stmt->close();
                        
                        // Hash the password
                        $password_hash = password_hash($password, PASSWORD_DEFAULT);
                        
                        // Update agent password
                        $update_stmt = $conn->prepare("UPDATE agents SET password_hash = ? WHERE contact = ?");
                        $update_stmt->bind_param("ss", $password_hash, $contact);
                        
                        if ($update_stmt->execute()) {
                            $message = "Password set successfully for agent: " . htmlspecialchars($agent['name']);
                        } else {
                            $error = 'Failed to set password. Please try again.';
                        }
                        
                        $update_stmt->close();
                    } else {
                        $error = 'Agent not found with contact: ' . htmlspecialchars($contact);
                    }
                    
                    $check_stmt->close();
                    $conn->close();
                }
            }
            
            // Display all agents
            $conn = getDBConnection();
            $agents_stmt = $conn->prepare("SELECT id, name, contact, password_hash FROM agents ORDER BY name");
            $agents_stmt->execute();
            $agents = $agents_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
            $agents_stmt->close();
            $conn->close();
            ?>
            
            <?php if ($error): ?>
                <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            
            <?php if ($message): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>
            
            <form method="POST" action="set_agent_password.php">
                <div class="form-group">
                    <label for="contact">Agent Contact (Phone Number)</label>
                    <input 
                        type="tel" 
                        id="contact" 
                        name="contact" 
                        placeholder="e.g., +254712345678" 
                        required
                        list="agent-contacts"
                        value="<?php echo htmlspecialchars($_POST['contact'] ?? ''); ?>"
                    >
                    <datalist id="agent-contacts">
                        <?php foreach ($agents as $agent): ?>
                            <option value="<?php echo htmlspecialchars($agent['contact']); ?>">
                                <?php echo htmlspecialchars($agent['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </datalist>
                </div>
                
                <div class="form-group">
                    <label for="password">New Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        placeholder="Enter password (min 6 characters)" 
                        required
                        minlength="6"
                    >
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">Confirm Password</label>
                    <input 
                        type="password" 
                        id="confirm_password" 
                        name="confirm_password" 
                        placeholder="Re-enter password" 
                        required
                        minlength="6"
                    >
                </div>
                
                <button type="submit" class="btn">Set Password</button>
            </form>
            
            <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid var(--gray-200);">
                <h3 style="font-size: 14px; margin-bottom: 10px; color: var(--gray-700);">Existing Agents</h3>
                <div style="max-height: 200px; overflow-y: auto;">
                    <?php foreach ($agents as $agent): ?>
                        <div style="padding: 8px; margin-bottom: 5px; background: var(--gray-50); border-radius: 6px; font-size: 13px;">
                            <strong><?php echo htmlspecialchars($agent['name']); ?></strong><br>
                            <span style="color: var(--gray-600);"><?php echo htmlspecialchars($agent['contact']); ?></span>
                            <span style="float: right; <?php echo !empty($agent['password_hash']) ? 'color: var(--success-color);' : 'color: var(--danger-color);'; ?>">
                                <?php echo !empty($agent['password_hash']) ? '✓ Has Password' : '✗ No Password'; ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <div style="margin-top: 20px; padding: 10px; background: #fff3cd; border-radius: 6px; font-size: 12px; color: #856404;">
                <strong>⚠️ Security Warning:</strong> Delete or restrict access to this file after setting up passwords!
            </div>
        </div>
    </div>
</body>
</html>